#ifndef AHLGREN_ENUMERATE_SEARCH
#define AHLGREN_ENUMERATE_SEARCH

#include "program.h"
#include "fitness.h"
#include "propositional.h" // bitstring
#include "bottom.h"


namespace lp {

	stat Program::generalize_enumerate(const sign_t& sign)
	{
		// Cache parameters
		const int param_csize = params.force_int(parameters::csize);
		const int param_noise = params.force_int(parameters::noise);
		const bool param_prune_consistent = params.is_set(parameters::prune_consistent);
		const bool param_enforce_io_spec = params.is_set(parameters::enforce_io_spec);
		const bool param_max_nodes = params.is_set(parameters::max_nodes);

		// Pick fitness function
		auto fitness = pick_fitness_function<bsf_type>(params);
		if (!fitness) {
			// Set default: lex
			fitness.reset( new fitness_lex<bsf_type>(params.force_int(parameters::terminate)) );
		}

		bitstring bs; // candidate as bitstring
		int bs_ones; // current number of ones
		std::set<bitstring> pruned; // all candidates below these ones are ignored
		int bot_size; // bottom size
		// Store mode I/O map
		io_map iomap;
		
		auto init_search = [&]
		(Program& thy, Functor& bottom, int, const std::vector<Mode>& modes, 
			std::list<clause>& pex, std::list<clause>& nex, Constraints&, stat& sts) -> bsf_type
		{
			fitness->init(bottom,modes,thy,pex,nex,sts[stat::exc]);
			// Increase bottom clause count
			++sts[stat::bottomc];
			// Compute and store mode structure
			if (param_enforce_io_spec) iomap = make_io_map(bottom,modes);
			pruned.clear();
			// Set start state
			bot_size = bottom.body_size();
			bs = bitstring(bot_size,0); // top element
			bs_ones = 0;
			return bsf_type();
		};

		auto find_candidate = [&]
		(Program& thy, std::list<clause>& pex, std::list<clause>& nex, Functor& bottom, int bsize, const std::vector<Mode>& modes, 
			const bsf_type& bsf, Constraints&, stat& sts, deadline_t deadline) -> bsf_type
		{
			bsf_type sol;
			const int MAX_ITER = std::numeric_limits<int>::max(); //50000;
			bool finished = false;
			int k = 0;
			for ( ; k < MAX_ITER && !finished; ++k) {
				// Have we finished the search?
				if (bs_ones > param_csize || bs_ones > bot_size) {
					throw search_completed();
				}
				// Check Time Out
				if (std::chrono::steady_clock::now() > deadline) {
					throw time_out();
				}

				// Create solution from bitstring
				mask_bottom(bottom,bs);
				sol = bs;

				// Update bitstring
				if (!std::next_permutation(bs.begin(),bs.end(),[](int x, int y){return x > y;})) {
					// Shift to next level by replacing 0 with 1
					++bs_ones;
					if (bs_ones <= bot_size && bs_ones <= param_csize) {
						fill_n(bs.begin(),bs_ones,1);
						fill(bs.begin()+bs_ones,bs.end(),0);
					} // else: finished search
				}

				// Check mode conformance
				if (std::any_of(pruned.begin(),pruned.end(),[&](const bitstring& f){return is_subset(f,sol.mask());})) {
					// Ignore pruned solution
				} else if (param_enforce_io_spec && !is_mode_conformant(bottom,thy.params,iomap)) {
					// Solution violates modes, tag as invalid
					sol.set_invalid();
					++sts[stat::invalid];
					finished = true;
				} else {
					++sts[stat::fitc];
					fitness->evaluate(bottom,sol,thy,pex,nex,bsf,sts[stat::exc],deadline);
					if (sol.is_consistent(param_noise)) {
						++sts[stat::conc];
						if (param_prune_consistent) {
							pruned.insert(sol.mask());
						}
					} else {
						++sts[stat::inconc];
					}
					finished = true;
				}
			} // until we produce a valid candidate

			if (k >= MAX_ITER) {
				std::cerr << "WARNING: enumeration gave up finding candidate after " << MAX_ITER << " attempts\n";
				throw search_aborted();
			}

			DEBUG_INFO(cout << "Sampled node: " << sol << "\n");
			return sol;
		};

		return this->generalize(sign,init_search,find_candidate);
	}
}


#endif

